<?php
declare(strict_types=1);

function app_config(): array {
    $path = __DIR__ . '/../config/config.php';
    if (!file_exists($path)) {
        http_response_code(500);
        die("Missing config/config.php. Run installer first.");
    }
    $cfg = require $path;
    if (!is_array($cfg)) {
        http_response_code(500);
        die("Invalid config format.");
    }
    return $cfg;
}

function db_conn(array $cfg): mysqli {
    mysqli_report(MYSQLI_REPORT_OFF);
    $conn = @new mysqli($cfg['db_host'], $cfg['db_user'], $cfg['db_pass'], $cfg['db_name']);
    if ($conn->connect_error) {
        http_response_code(500);
        die("Database connection failed.");
    }
    $conn->set_charset('utf8mb4');
    return $conn;
}

function get_license_key(mysqli $conn): string {
    $res = $conn->query("SELECT license_key FROM settings WHERE id=1 LIMIT 1");
    if (!$res) return '';
    $row = $res->fetch_assoc();
    return trim((string)($row['license_key'] ?? ''));
}

function license_cache_path(): string {
    return __DIR__ . '/license_cache.json';
}

function read_cache(): array {
    $p = license_cache_path();
    if (!file_exists($p)) return [];
    $raw = file_get_contents($p);
    $j = json_decode($raw ?: '', true);
    return is_array($j) ? $j : [];
}

function write_cache(array $data): void {
    file_put_contents(license_cache_path(), json_encode($data, JSON_PRETTY_PRINT), LOCK_EX);
}

function verify_license_online(array $cfg, string $licenseKey): array {
    $domain = $_SERVER['HTTP_HOST'] ?? 'unknown';

    if (!function_exists('curl_init')) {
        return ['status' => 'error', 'message' => 'cURL not enabled'];
    }

    $ch = curl_init($cfg['license_url']);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => [
            'license' => $licenseKey,
            'domain'  => $domain,
        ],
        CURLOPT_TIMEOUT => 8,
        CURLOPT_CONNECTTIMEOUT => 5,
    ]);

    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($resp === false || $code < 200 || $code >= 300) {
        return ['status' => 'error', 'message' => $err ?: ('HTTP ' . $code)];
    }

    $json = json_decode($resp, true);
    if (!is_array($json)) return ['status' => 'error', 'message' => 'Invalid JSON response'];
    return $json;
}

function enforce_license(): void {
    $cfg = app_config();
    $conn = db_conn($cfg);

    $key = get_license_key($conn);
    if ($key === '') {
        http_response_code(403);
        die("License missing. Please add your license key.");
    }

    $ttl = (int)($cfg['license_cache_ttl'] ?? 86400);
    $cache = read_cache();

    $domain = $_SERVER['HTTP_HOST'] ?? 'unknown';
    $cacheKey = hash('sha256', $key . '|' . $domain);

    $now = time();
    if (isset($cache[$cacheKey]) && is_array($cache[$cacheKey])) {
        $c = $cache[$cacheKey];
        if (($c['status'] ?? '') === 'valid' && (int)($c['expires_at'] ?? 0) > $now) {
            return;
        }
    }

    $result = verify_license_online($cfg, $key);

    if (($result['status'] ?? '') === 'valid') {
        $cache[$cacheKey] = [
            'status' => 'valid',
            'expires_at' => $now + max(300, $ttl),
            'checked_at' => $now,
        ];
        write_cache($cache);
        return;
    }

    http_response_code(403);
    $status = htmlspecialchars((string)($result['status'] ?? 'invalid'), ENT_QUOTES, 'UTF-8');
    die("License check failed: {$status}");
}
